% This m-file is for calculating the 3D (emission or excitation) PSF for
% a single dipole emitter with fixed or free dipole axis using the
% CZT-function with arbitrary windows and discretization in real and
% spatial frequency space.
%
% The axial coordinate is the z-stage position, increasing with distance
% from the objective to the cover slip. The nominal z-stage position is
% found for a given set of refractive index values and a given depth of
% the image plane away from the cover slip into the medium by minimizing
% the rms wavefront aberration.
%
% Sjoerd Stallinga, TU Delft

% (C) Copyright 2018
% All rights reserved
% Department of Imaging Physics
% Faculty of Applied Sciences
% Delft University of Technology
% Delft, The Netherlands   

close all;
clear all;

% run('C:\Program Files\DIPimage 2.8.1\dipstart.m')

save_flg = 1;
plot_flg = 1;

%%
% set parameters
parameters = set_parameters;

% number of configurations
Ncfg = 1;

% emitter position (in nm) with z-position from image plane
xemit = 0*parameters.pixelsize*(2*rand(Ncfg,1)-1);
yemit = 0*parameters.pixelsize*(2*rand(Ncfg,1)-1);

allPSFs = zeros(parameters.Mx,parameters.My,parameters.Mz,Ncfg);

tic
for jj = 1:Ncfg
    
    parameters.xemit = xemit(jj);
    parameters.yemit = yemit(jj);
    
    % calculate pupil matrix
    [XPupil,YPupil,wavevector,wavevectorzmed,Waberration,PupilMatrix] = get_pupil_matrix(parameters);
    
    % calculate field matrix for focal stack
    [XImage,YImage,ZImage,FieldMatrix] = get_field_matrix(PupilMatrix,wavevector,wavevectorzmed,parameters);
    
    % calculation of (free dipole) PSF for the focal stack
    % [FreePSF,FixedPSF] = get_psfs(FieldMatrix,parameters);
    allPSFs(:,:,:,jj) = get_freedipolepsf(FieldMatrix,parameters);
    
%     calculation of through-focus OTF for the focal stack
%     [XSupport,YSupport,OTFthroughfocus] = get_otf(allPSFs,XImage,YImage,parameters);
    
%     calculation of 3D-OTF
%     [ZSupport,OTF3D] = get_otf3d(OTFthroughfocus,ZImage,parameters);
            
    % 3D convolution of the PSFs and derivatives with a bead
    if isfield(parameters,'bead')
        if parameters.bead == true
            bead = create3DBead(parameters);
            allPSFs = convn(bead,allPSFs,'same');
            [Mx,My,Mz] = size(allPSFs);
        end
    end
  
end
toc

parameters.xemit = xemit;
parameters.yemit = yemit;

%%
% make plots

if plot_flg && Ncfg == 1
    
    fs = 13;
    fn = 'Arial';
    lw = 1.5;
    
    diffunitxy = parameters.lambda/parameters.NA;
    diffunitz = parameters.lambda/(parameters.refimm-sqrt(parameters.refimm^2-parameters.NA^2));
    
    jz = ceil(parameters.Mz/2);
    tempim = squeeze(allPSFs(:,:,jz));
    radialmeanPSF = im2mat(radialmean(tempim,[],1,parameters.xrange-parameters.pixelsize/2));
    radialmeanPSF = radialmeanPSF/max(radialmeanPSF);
    FWHM = 2*(sum(double(radialmeanPSF>0.5))-0.5)*parameters.pixelsize;
    fprintf('FWHM of PSF = %6.1f nm\n',FWHM)
    
    if (parameters.Mz>1)

        figure;
        ximagelin = squeeze(XImage(:,1))';
        yimagelin = squeeze(YImage(1,:));
        sliceposx = parameters.xrange/parameters.Mx;
        sliceposy = parameters.yrange/parameters.My;
        sliceposz = 0;
        slice(yimagelin,ximagelin,ZImage,allPSFs,sliceposx,sliceposy,sliceposz);
        
        axis square;
        shading flat;
        box on
        grid off
        
        view(90,0)
        
        xlim([-2e3 2e3]);    %   ylim([-parameters.xrange,parameters.xrange]);
        ylim([-2e3 2e3]);    %   xlim([-parameters.yrange,parameters.yrange]);
        
        %     title('PSF','Interpreter','LaTex');
        ylabel('$x$ [nm]','Interpreter','LaTex');
        xlabel('$y$ [nm]','Interpreter','LaTex');
        zlabel('${\Delta}z$ [nm]','Interpreter','LaTex');
        
        set(gca,...
            'color','none','Units','normalized','FontUnits','points',...
            'FontWeight','normal','TickLabelInterpreter','LaTex',...
            'FontSize',fs,'FontName',fn)
        
        figure;
        ximagelin = squeeze(XImage(:,1))';
        yimagelin = squeeze(YImage(1,:));
        sliceposx = parameters.xrange/parameters.Mx;
        sliceposy = parameters.yrange/parameters.My;
        sliceposz = 0;
        slice(yimagelin,ximagelin,ZImage,allPSFs,sliceposx,sliceposy,sliceposz);
        
        axis square;
        shading flat;
        box on
        grid off
        
        view(0,0)
        
        xlim([-2e3 2e3]);    %   ylim([-parameters.xrange,parameters.xrange]);
        ylim([-2e3 2e3]);    %   xlim([-parameters.yrange,parameters.yrange]);
        
        %     title('PSF','Interpreter','LaTex');
        ylabel('$x$ [nm]','Interpreter','LaTex');
        xlabel('$y$ [nm]','Interpreter','LaTex');
        zlabel('${\Delta}z$ [nm]','Interpreter','LaTex');
        
        set(gca,...
            'color','none','Units','normalized','FontUnits','points',...
            'FontWeight','normal','TickLabelInterpreter','LaTex',...
            'FontSize',fs,'FontName',fn)

        figure;
        box on
        PSF_onaxis = squeeze(allPSFs(round(parameters.Mx/2),round(parameters.My/2),:));
        plot(ZImage,PSF_onaxis./max(PSF_onaxis),'LineWidth',lw);
        
        xlabel('${\Delta}z$ [nm]','Interpreter','LaTex');
        ylabel('On axis PSF','Interpreter','LaTex');
        
        set(gca,...
            'color','none','Units','normalized','FontUnits','points',...
            'FontWeight','normal','TickLabelInterpreter','LaTex',...
            'FontSize',fs,'FontName',fn)
        
        
        % radial avgerage focused PSF
        positionvec = (0:(length(radialmeanPSF)-1))*parameters.pixelsize;
        
        figure
        plot(positionvec,radialmeanPSF,'LineWidth',lw);
        
        box on
        hold on
        
        xlim([0 1000]); % xlim([0 3*parameters.lambda/parameters.NA])
        
        xlabel('Radial position [nm]','Interpreter','LaTex');
        ylabel('Radial average PSF ','Interpreter','LaTex');
        
        set(gca,...
            'color','none','Units','normalized','FontUnits','points',...
            'FontWeight','normal','TickLabelInterpreter','LaTex',...
            'FontSize',fs,'FontName',fn)
        
        
        %   figure;
        %   XYSupport = squeeze(XSupport(1,:));
        %   sliceposx = 0.0;
        %   sliceposy = 0.0;
        %   sliceposz = 0.0;
        %   slice(diffunitxy*XYSupport,diffunitxy*XYSupport,ZImage,abs(OTFthroughfocus),sliceposx,sliceposy,sliceposz);
        %   title('through-focus MTF');
        %   axis square;
        %   shading flat;
        %   xlim([-2,2]);
        %   ylim([-2,2]);
        %   colorbar;
        %   xlabel('q_x [NA/\lambda]');
        %   ylabel('q_y [NA/\lambda]');
        %   zlabel('{\Delta}z_{stage} [nm]');
        %
        %   figure;
        %   XYSupport = squeeze(XSupport(1,:));
        %   sliceposx = 0.0;
        %   sliceposy = 0.0;
        %   sliceposz = 0.0;
        %   slice(diffunitxy*XYSupport,diffunitxy*XYSupport,diffunitz*ZSupport,abs(OTF3D),sliceposx,sliceposy,sliceposz);
        %   title('3D MTF');
        %   axis square;
        %   shading flat;
        %   xlim([-2,2]);
        %   ylim([-2,2]);
        %   zlim([-3,3]);
        %   colorbar;
        %   xlabel('q_x [NA/\lambda]');
        %   ylabel('q_y [NA/\lambda]');
        %   zlabel('q_z [n-(n^{2}-NA^{2})^{1/2})/\lambda]');
        
    else
            figure;
            surf(XImage,YImage,allPSFs);
            title('PSF');
            axis square;
            shading flat;
            xlim([-parameters.xrange,parameters.xrange]);
            ylim([-parameters.yrange,parameters.yrange]);
            colorbar;
            xlabel('x [nm]');
            ylabel('y [nm]');

            figure;
            surf(diffunitxy*XSupport,diffunitxy*YSupport,abs(OTFthroughfocus));
            title('MTF');
            axis square;
            shading flat;
            xlim([-2,2]);
            ylim([-2,2]);
            colorbar;
            xlabel('q_x [NA/\lambda]');
            ylabel('q_y [NA/\lambda]');
        
        positionvec = (0:(length(radialmeanPSF)-1))*parameters.pixelsize;
        
        figure
        plot(positionvec,radialmeanPSF,'LineWidth',lw);
        
        box on
        hold on
        
        xlim([0 1000]); % xlim([0 3*parameters.lambda/parameters.NA])
        
        xlabel('Radial position [nm]','Interpreter','LaTex');
        ylabel('Radial average PSF ','Interpreter','LaTex');
        
        set(gca,...
            'color','none','Units','normalized','FontUnits','points',...
            'FontWeight','normal','TickLabelInterpreter','LaTex',...
            'FontSize',fs,'FontName',fn)
        
    end
end

dipshow(allPSFs)





